import 'dart:convert';

import 'package:http/browser_client.dart';
import 'package:photogram_admin_cp/import/core.dart';
import 'package:photogram_admin_cp/import/data.dart';

class ApiProvider {
  late final Uri _apiUrl;
  late final String _apiVersion;

  final _client = BrowserClient();

  final _attachments = <String, FileDetails>{};

  /*
  |--------------------------------------------------------------------------
  | constructor, setup end point details
  |--------------------------------------------------------------------------
  */

  ApiProvider({
    required String apiUrl,
    required String apiVersion,
  }) {
    _apiUrl = Uri.parse(apiUrl);
    _apiVersion = apiVersion;
  }

  /*
  |--------------------------------------------------------------------------
  | helpers
  |--------------------------------------------------------------------------
  */

  void addAttachment(String key, FileDetails fileDetails) {
    _attachments[key] = fileDetails;
  }

  void clearAttachments() {
    _attachments.clear();
  }

  bool isAttachmentKeyAvailable(String key) => _attachments.containsKey(key);

  /*
  |--------------------------------------------------------------------------
  | request processor original:
  |--------------------------------------------------------------------------
  */

  Future<ResponseModel> _makeRequest({
    required String requestType,
    required Map<String, dynamic> body,
  }) async {
    AppLogger.info("ApiProvider: Request($requestType) => ", error: json.encode(body));

    var jsonMap = <String, dynamic>{};

    try {
      // _client.withCredentials = true;

      var response = await _client.post(_apiUrl,
          body: json.encode(
            {
              RequestTable.apiVersion: _apiVersion,
              RequestTable.reqType: requestType,
              RequestTable.payload: body,
            },
          ));

      jsonMap = Map<String, dynamic>.from(jsonDecode(utf8.decode(response.bodyBytes)));
    } catch (e) {
      AppLogger.exception(e);
    }

    return ResponseModel.fromJson(jsonMap);
  }

  /*
  |--------------------------------------------------------------------------
  | prepared request:
  |--------------------------------------------------------------------------
  */

  Future<ResponseModel> preparedRequest({
    required String requestType,
    required Map<String, dynamic> requestData,
  }) async =>
      await _makeRequest(
        requestType: requestType,
        body: requestData,
      );

  /*
  |--------------------------------------------------------------------------
  | auth session request:
  |--------------------------------------------------------------------------
  */

  Future<ResponseModel> adminAuthSession() async => await _makeRequest(
        requestType: REQ_TYPE_ADMIN_SESSION,
        body: {},
      );
}
